<?php

namespace Mnv\Core\JSONParser\Writer;

/**
 *Базовый класс JSONParser для написания электронной таблицы
 *
 * @package JSONParser
 */
abstract class BaseWriter implements IWriter
{
    /**
     * Хранит табличные данные
     *
     * @access  protected
     * @var     array
     */
    protected array $table_arr;

    /**
     * Определяет тип содержимого для HTTP-заголовка
     *
     * @access  protected
     * @var     string
     */
    protected string $content_type = 'application/json';

    /**
     * Определяет расширение файла, которое будет использоваться при сохранении файла
     *
     * @access  protected
     * @var     string
     */
    protected string $file_extension = 'json';

    /**
     * @return  void
     */
    public function __construct()
    {
        $this->table_arr = array();
    }

    /**
     * Добавление данных строки в таблицу
     *
     * @param   array   $values An array contains ordered value for every cell
     * @param   bool    Check if row goes at the beginning or end of array
     * @return  void
     */


    /**
     *  Редактирование данных строки в таблицу
     *
     * @param $index
     * @param $value
     * @return bool
     */
    public function editRow($index, $values): bool
    {
        if (!$this->isRowExists($index)) {
            return false;
        }

        if (!is_array($values)) {
            $values = array($values);
        }

        foreach ($this->table_arr as $key => $item) {
            if ($key == $index)
                $this->table_arr[$key] = $values;
        }
        return true;
    }

    /**
     * Добавление данных строки в таблицу
     *
     * @param   array   $values Массив содержит упорядоченное значение для каждой ячейки
     * @param   bool    Проверьте, находится ли строка в начале или в конце массива
     * @return  void
     */
    public function addArray(array $values, $end = TRUE)
    {
        if ($end) {
            array_push($this->table_arr, $values);
            return;
        }
        array_unshift($this->table_arr, $values);
    }

    /**
     * Добавление данных строки в таблицу
     *
     * @param   array   $values Массив содержит упорядоченное значение для каждой ячейки
     * @param   bool    Проверьте, находится ли строка в начале или в конце массива
     * @return  void
     */
    public function addRow($values, $end = TRUE)
    {
        if (!is_array($values)) {
            $values = array($values);
        }
        if ($end) {
            array_push($this->table_arr, $values);
            return;
        }
        array_unshift($this->table_arr, $values);
    }

    public function deleteRow($index): bool
    {
        if (!$this->isRowExists($index)) {
            return false;
        }
        foreach ($this->table_arr as $key => $item) {
            if ($key == $index)
                unset($this->table_arr[$key]);
        }
        return true;
    }

    /**
     * Получить содержимое документа в виде строки
     *
     * @return  string  Content of document
     */
//    public function saveString()
//    {
//        $content = '';
//        foreach ($this->table_arr as $row) {
//            foreach ($row as $cell) {
//                $content .= $cell.'\t';
//            }
//            $content .= '\n';
//        }
//        return $content;
//    }

    /**
     * Экспорт документа
     *
     * @param   string  $filename   Name for the saved file (extension will be set automatically)
     * @param   string  $target     Save location
     * @return  void
     */
    public function saveFile($filename, $target = NULL)
    {
        if (!isset($filename)) {
            $filename = date('YmdHis');
        }

        if (!isset($target)) {
            // Написать выход на браузер
            $target = 'php://output';

            // Установить заголовок HTTP ответ
            header('Content-Type: ' . $this->content_type);
            header('Content-Disposition: attachment; filename=' . $filename . '.' . $this->file_extension);
        }

        $fp = fopen($target, 'w');
        fwrite($fp, $this->saveString());
        fclose($fp);

        if ($target == 'php://output') {
            // Так как не должно быть данных ниже
            exit();
        }
    }

    /**
     * Установить табличные данные
     *
     * @param   array   $values Массив содержит упорядоченное значение массивов для всех полей
     * @return  void
     */
    public function setData($values)
    {
        if (!is_array($values)) {
            $values = array($values);
        }
        $this->table_arr = $values;
    }

    /**
     * Проверить, существует ли указанная строка
     *
     * @param    int   Row number
     * @return   bool
     */
    public function isRowExists($row_num): bool
    {
        return array_key_exists($row_num, $this->table_arr);
    }
}